#include "animation.h"
/*
 Walking:      [0 1 2 3] [0 1 2 3] [0 1 2 3] [0 1 2 3] [0 1 2 3] [0 1 2 3] [0 1 2 3] [0 1 2 3]
               0         1         2         3*        4         3(5)      2(6)      1(7)
 Turning:      [0 1 2 3] [0 1 2 3] [0 1 2 3] [0 1 2 3]
               0         1         0(2)      copy(3)
 Jumping:      [0 1 2 3] [0 1 2 3] [0 1 2 3] [0 1 2 3] [0 1 2 3] [0 1 2 3] [0 1 2 3]
               0(7)      1(8)      2(9)      3(10*)    2         1         0
 JumpingDone:  Frames 0/1/2 of Jumping in reverse
 Dying         [0 1 2 3] [0 1 2 3] [0 1 2 3] [0 1 2 3]
               11        12        13        14*
 Attacking:    [0 1 2 3] [0 1 2 3] [0 1 2 3] [0 1 2 3] [0 1 2 3] [0 1 2 3] [0 1 2 3] [0 1 2 3]
               2         15        16        17        18        17        16        15
 Sitting:      [0 1 2 3] [0 1 2 3] [0 1 2 3] [0 1 2 3]
               19        20        21        22*
 *hold

 [00 01 02 03 04] [05 06] [07 08 09 10] [11 12 13 14] [15 16 17 18] [19 20 21 22]
 walk             turn    jump          die           attack        sit

*/
static void _get_dimensions_from_animation(LandSpriteType * st, LandAnimation * a, int safe_zone);
static void _add_background(LandImage * parent, int x, int y, int h);
void get_animation_frame(Player * p, int * f, int * flip) {
    for (unsigned int d = 0; d < 2; d += 1) {
        if (p->anim == StandingRight + d) {
            * f = 2;
            * flip = d;
        }
        else if (p->anim == WalkingRight + d) {
            * f = land_div(p->anim_t, 4);
            if (* f >= 5) {
                * f = 8 - * f;
            }
            * flip = d;
        }
        else if (p->anim == AttackingRight + d) {
            * f = land_div(p->anim_t, 4);
            if (* f >= 5) {
                * f = 8 - * f;
            }
            * f = 14 + * f;
            if (* f == 14) {
                * f = 2;
            }
            * flip = d;
        }
        else if (p->anim == TurningRight + d) {
            * f = land_div(p->anim_t, 4);
            if (* f <= 1) {
                * f = 5 + * f;
                * flip = 1 - d;
            }
            else {
                * f = 8 - * f;
                * flip = d;
            }
        }
        else if (p->anim == JumpingRight + d) {
            * f = 7 + land_div(p->anim_t, 4);
            * flip = d;
        }
        else if (p->anim == JumpingDoneRight + d) {
            * f = 9 - land_div(p->anim_t, 4);
            * flip = d;
        }
        else if (p->anim == DyingRight + d) {
            * f = 11 + land_div(p->anim_t, 4);
            * flip = d;
        }
        else if (p->anim == ArisingRight + d) {
            * f = 13 - land_div(p->anim_t, 4);
            * flip = d;
        }
        else if (p->anim == SittingRight + d) {
            * f = 19 + land_div(p->anim_t, 4);
            if (* f > 22) {
                * f = 22;
            }
            * flip = d;
        }
        else if (p->anim == GettingUpRight + d) {
            * f = 21 - land_div(p->anim_t, 4);
            * flip = d;
        }
    }
}
// use right left if kx<>0, or if kx==0 use p.eft
void set_wanted_animation(Player * p, AnimationState anim2, int kx) {
    if (anim2 == SittingRight) {
        if (! player_have_animation_frame(p, 19)) {
            return ;
        }
    }
    if (anim2 == AttackingRight) {
        if (! player_have_animation_frame(p, 15)) {
            return ;
        }
    }
    if (anim2 == DyingRight) {
        if (! player_have_animation_frame(p, 11)) {
            return ;
        }
    }
    if (anim2 == JumpingRight) {
        if (! player_have_animation_frame(p, 8)) {
            return ;
        }
    }
    if (kx == 0) {
        if (p->left) {
            anim2 += 1;
        }
    }
    else if (kx < 0) {
        anim2 += 1;
    }
    p->anim_t += 1;
    if (p->anim == WalkingRight || p->anim == WalkingLeft) {
        // animate double fast when we are preparing to jump
        if ((anim2 == JumpingRight && p->anim == WalkingRight) || (anim2 == JumpingLeft && p->anim == WalkingLeft)) {
            if (p->anim_t != 5 * 4) {
                p->anim_t += 1;
            }
        }
        if (p->anim_t >= 8 * 4) {
            p->anim_t -= 8 * 4;
        }
    }
    else if (p->anim == JumpingRight || p->anim == JumpingLeft) {
        if (p->anim_t == 4 * 4) {
            // hold frame
            p->anim_t -= 1;
        }
    }
    else if (p->anim == DyingRight || p->anim == DyingLeft) {
        if (p->anim_t == 4 * 4) {
            // hold frame
            p->anim_t -= 1;
        }
    }
    else if (p->anim == AttackingRight || p->anim == AttackingLeft) {
        if (p->anim_t == 4 * 8) {
            p->anim_t = 0;
        }
    }
    if (p->anim == anim2) {
        return ;
    }
    for (unsigned int d = 0; d < 2; d += 1) {
        unsigned int o = 1 - d;
        if (p->anim == StandingRight + d) {
            if (anim2 == WalkingRight + d || anim2 == JumpingRight + d || anim2 == AttackingRight + d) {
                p->anim = WalkingRight + d;
                p->anim_t = 3 * 4;
            }
            else if (anim2 == WalkingRight + o || anim2 == StandingRight + o || anim2 == JumpingRight + o || anim2 == DyingRight + o) {
                p->anim = TurningRight + o;
                p->anim_t = 0;
            }
            else if (anim2 == DyingRight + d) {
                if (p->anim_t >= 1 * 4) {
                    // play the complete frame
                    p->anim = anim2;
                    p->anim_t = 0;
                }
            }
            else if (anim2 == SittingRight + d) {
                play(BoredAudio, 1, 0, 1);
                p->anim = anim2;
                p->anim_t = 0;
            }
        }
        else if (p->anim == WalkingRight + d) {
            if (anim2 == StandingRight + d || anim2 == DyingRight + d) {
                if ((p->anim_t == 2 * 4 /* standing frame */ || p->anim_t == 4 * 4 /* backstep from frame 3 to frame 2 */ || p->anim_t == 6 * 4 /* standing frame */ || p->anim_t == 0 * 4)) {
                    // forward step from frame 1 to frame 2
                    p->anim = StandingRight + d;
                    p->anim_t = 0;
                }
            }
            else if (anim2 == WalkingRight + o || anim2 == StandingRight + o || anim2 == JumpingRight + o || anim2 == DyingRight + o) {
                if (p->anim_t == 3 * 4 || p->anim_t == 7 * 4) {
                    p->anim = TurningRight + o;
                    p->anim_t = 0;
                }
            }
            else if (anim2 == JumpingRight + d) {
                if (p->anim_t == 5 * 4) {
                    p->anim = anim2;
                    p->anim_t = 0;
                }
            }
            else if (anim2 == AttackingRight + d) {
                if (p->anim_t == 2 * 4 || p->anim_t == 4 * 4 || p->anim_t == 6 * 4 || p->anim_t == 0 * 4) {
                    p->anim = anim2;
                    p->anim_t = 0;
                }
                else if (p->anim_t == 3 * 4 || p->anim_t == 7 * 4) {
                    p->anim = anim2;
                    p->anim_t = 1 * 4;
                }
            }
        }
        else if (p->anim == TurningRight + d) {
            if (anim2 == WalkingRight + d || anim2 == StandingRight + d || anim2 == JumpingRight + d || anim2 == DyingRight + d || anim2 == AttackingRight + d) {
                if (p->anim_t >= 4 * 4) {
                    if (anim2 == JumpingRight + d) {
                        p->anim = WalkingRight + d;
                    }
                    else {
                        p->anim = anim2;
                    }
                    p->anim_t = 2 * 4;
                }
            }
            else if (anim2 == WalkingRight + o || anim2 == StandingRight + o || anim2 == JumpingRight + o || anim2 == DyingRight + o || anim2 == AttackingRight + o) {
                if (p->anim_t == 1 * 4) {
                    p->anim = WalkingRight + o;
                    p->anim_t = 2 * 4;
                }
                else if (p->anim_t == 2 * 4) {
                    p->anim = TurningRight + o;
                    p->anim_t = 3 * 4;
                }
                else if (p->anim_t == 3 * 4) {
                    p->anim = TurningRight + o;
                    p->anim_t = 2 * 4;
                }
                else if (p->anim_t >= 4 * 4) {
                    p->anim = TurningRight + o;
                    p->anim_t = 1 * 4;
                }
            }
        }
        else if (p->anim == JumpingRight + d) {
            p->anim = JumpingDoneRight + d;
            p->anim_t = 0;
        }
        else if (p->anim == JumpingDoneRight + d) {
            if (p->anim_t == 3 * 4) {
                p->anim = WalkingRight + d;
                p->anim_t = 4 * 4;
            }
        }
        else if (p->anim == DyingRight + d) {
            p->anim = ArisingRight + d;
            p->anim_t = 0;
        }
        else if (p->anim == ArisingRight + d) {
            if (p->anim_t == 3 * 4) {
                p->anim_t = 0;
                p->anim = anim2;
            }
        }
        else if (p->anim == AttackingRight + d) {
            if (p->anim_t == 0) {
                p->anim = anim2;
            }
        }
        else if (p->anim == SittingRight + d) {
            p->anim = GettingUpRight + d;
            p->anim_t = 0;
        }
        else if (p->anim == GettingUpRight + d) {
            if (p->anim_t == 3 * 4) {
                p->anim = StandingRight + d;
                p->anim_t = 0;
            }
        }
    }
}
LandSpriteType* get_animation_type(str name, bool is_player) {
    Game * game = game_global();
    if (! game->animations) {
        game->animations = land_hash_new();
    }
    LandSpriteType * st = land_hash_get(game->animations, name);
    if (st) {
        return st;
    }
    char * aname = land_strdup(name);
    land_append(& aname, "%s", "_%04d.png");
    if (is_player) {
        LandAnimation * anim = land_atlas_animation_create_fixed_count(game->atlas, aname, LAND_IMAGE_CENTER, 23);
        land_free(aname);
        st = land_spritetype_animation_new(anim, NULL, 0 /* mask */, 0 /* n */);
        _get_dimensions_from_animation(st, anim, 4);
        if (land_equals(name, "rootspell")) {
            st->y += st->h * 2;
            st->h += st->h * 2;
        }
    }
    else {
        LandAnimation * anim = land_atlas_animation_create(game->atlas, aname, LAND_IMAGE_CENTER);
        st = land_spritetype_animation_new(anim, NULL, 0 /* mask */, 0 /* n */);
        _get_dimensions_from_animation(st, anim, 4);
    }
    land_hash_insert(game->animations, name, st);
    if (is_player) {
        st->draw = player_draw;
    }
    else {
        LandSpriteTypeAnimation * at = (void *) st;
        at->auto_speed = 4;
    }
    return st;
}
static void _get_dimensions_from_animation(LandSpriteType * st, LandAnimation * a, int safe_zone) {
    /* get the smallest dimensions of the walking frames (0..4), flipped or not
     */
    float s = 0.5 /* sprites are globally drawn scaled 0.5 */;
    float maxl = - 1000;
    float minr = 1000;
    float maxt = - 1000;
    float minb = 1000;
    int f = 0;
    {
        LandArrayIterator __iter0__ = LandArrayIterator_first(a->frames);
        for (LandImage * im = LandArrayIterator_item(a->frames, &__iter0__); LandArrayIterator_next(a->frames, &__iter0__); im = LandArrayIterator_item(a->frames, &__iter0__)) {
            float lx = - im->x;
            float tx = - im->y;
            float rx = im->width - im->x;
            float by = im->height - im->y;
            if (lx > maxl) {
                maxl = lx;
            }
            if (tx > maxt) {
                maxt = tx;
            }
            if (rx < minr) {
                minr = rx;
            }
            if (by < minb) {
                minb = by;
            }
            if (- rx > maxl) {
                maxl = - rx /* flipped */;
            }
            if (- lx < minr) {
                minr = - lx /* flipped */;
            }
            f += 1;
            if (f == 5) {
                break;
            }
        }
    }
    st->w = (minr - maxl) * s - safe_zone * 2;
    st->x = - maxl * s - safe_zone;
    st->h = (minb - maxt) * s - safe_zone * 2;
    st->y = - maxt * s - safe_zone;
}
//print("%.0f %.0f %.0f %.0f", st.x, st.y, st.w, st.h)
static void _add_background(LandImage * parent, int x, int y, int h) {
    Game * game = game_global();
    if (! game->backgrounds) {
        game->backgrounds = land_array_new();
    }
    land_array_add(game->backgrounds, land_image_sub(parent, x * h, y * h, h, h));
}
LandImage* get_background(str key) {
    Game * game = game_global();
    if (key [0] == 'm') {
        int i = key [1] - '0';
        return land_array_get(game->backgrounds, 0 + i);
    }
    if (key [0] == 'e') {
        int i = key [1] - '0';
        return land_array_get(game->backgrounds, 4 + i);
    }
    if (key [0] == 'f') {
        int i = key [1] - '0';
        return land_array_get(game->backgrounds, 8 + i);
    }
    if (key [0] == 't') {
        int i = key [1] - '0';
        return land_array_get(game->backgrounds, 12 + i);
    }
    if (key [0] == 'c') {
        int i = key [1] - '0';
        return land_array_get(game->backgrounds, 20 + i);
    }
    return NULL;
}
void load_backgrounds(void) {
    LandImage * b1 = land_image_load("meadow.png");
    LandImage * b2 = land_image_load("forestedge.png");
    LandImage * b3 = land_image_load("forest.png");
    LandImage * b4 = land_image_load("tree.png");
    LandImage * b5 = land_image_load("clearing.png");
    int h = land_image_height(b1);
    _add_background(b1, 0, 0, h);
    _add_background(b1, 1, 0, h);
    _add_background(b1, 2, 0, h);
    _add_background(b1, 3, 0, h);
    _add_background(b2, 0, 0, h);
    _add_background(b2, 1, 0, h);
    _add_background(b2, 2, 0, h);
    _add_background(b2, 3, 0, h);
    _add_background(b3, 0, 0, h);
    _add_background(b3, 1, 0, h);
    _add_background(b3, 2, 0, h);
    _add_background(b3, 3, 0, h);
    _add_background(b4, 0, 0, h);
    _add_background(b4, 0, 1, h);
    _add_background(b4, 0, 2, h);
    _add_background(b4, 0, 3, h);
    _add_background(b4, 1, 0, h);
    _add_background(b4, 1, 1, h);
    _add_background(b4, 1, 2, h);
    _add_background(b4, 1, 3, h);
    _add_background(b5, 0, 0, h);
    _add_background(b5, 1, 0, h);
    _add_background(b5, 2, 0, h);
    _add_background(b5, 3, 0, h);
    land_image_blend_into(get_background("e3"), get_background("f3"), 0 /* left_x */, 0, 1 /* right_x */, 1);
    land_image_blend_into(get_background("e0"), get_background("f0"), 0 /* left_x */, 1, 1 /* right_x */, 0);
    land_image_blend_into(get_background("e1"), get_background("m1"),  /* left_x */0.5, 0, 1 /* right_x */, 1);
    land_image_blend_into(get_background("e2"), get_background("m2"), 0 /* left_x */, 1,  /* right_x */0.5, 0);
}
